/*
 * Decompiled with CFR 0.152.
 */
package info.ata4.io.buffer;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

public class ByteBufferUtils {
    private static final ByteBuffer EMPTY = ByteBuffer.allocate(0);
    private static final int DIRECT_THRESHOLD = 10240;

    private ByteBufferUtils() {
    }

    private static int truncateLength(FileChannel fc, int length) throws IOException {
        return (int)Math.min(length > 0 ? (long)length : fc.size(), Integer.MAX_VALUE);
    }

    public static ByteBuffer load(Path path, int offset, int length) throws IOException {
        try (FileChannel fc = FileChannel.open(path, StandardOpenOption.READ);){
            ByteBuffer bb;
            int size = ByteBufferUtils.truncateLength(fc, length);
            if (size > 10240) {
                bb = ByteBuffer.allocateDirect(size);
            } else {
                try {
                    bb = ByteBuffer.allocate(size);
                }
                catch (OutOfMemoryError ex) {
                    bb = ByteBuffer.allocateDirect(size);
                }
            }
            fc.position(offset);
            fc.read(bb);
            bb.flip();
            ByteBuffer byteBuffer = bb;
            return byteBuffer;
        }
    }

    public static ByteBuffer load(Path path) throws IOException {
        return ByteBufferUtils.load(path, 0, -1);
    }

    public static ByteBuffer load(List<Path> paths) throws IOException {
        ArrayList<ByteBuffer> bbs = new ArrayList<ByteBuffer>();
        for (Path path : paths) {
            bbs.add(ByteBufferUtils.openReadOnly(path));
        }
        return ByteBufferUtils.concat(bbs);
    }

    public static void save(Path path, ByteBuffer bb) throws IOException {
        try (FileChannel fc = FileChannel.open(path, StandardOpenOption.WRITE, StandardOpenOption.CREATE);){
            fc.truncate(bb.remaining());
            fc.write(bb);
        }
    }

    public static MappedByteBuffer openReadOnly(Path path, int offset, int length) throws IOException {
        try (FileChannel fc = FileChannel.open(path, StandardOpenOption.READ);){
            MappedByteBuffer mappedByteBuffer = fc.map(FileChannel.MapMode.READ_ONLY, offset, ByteBufferUtils.truncateLength(fc, length));
            return mappedByteBuffer;
        }
    }

    public static MappedByteBuffer openReadOnly(Path path) throws IOException {
        return ByteBufferUtils.openReadOnly(path, 0, -1);
    }

    public static MappedByteBuffer openReadWrite(Path path, int offset, int length) throws IOException {
        try (FileChannel fc = FileChannel.open(path, StandardOpenOption.READ, StandardOpenOption.WRITE, StandardOpenOption.CREATE);){
            MappedByteBuffer mappedByteBuffer = fc.map(FileChannel.MapMode.READ_WRITE, offset, ByteBufferUtils.truncateLength(fc, length));
            return mappedByteBuffer;
        }
    }

    public static MappedByteBuffer openReadWrite(Path path) throws IOException {
        return ByteBufferUtils.openReadWrite(path, 0, -1);
    }

    public static ByteBuffer getSlice(ByteBuffer bb, int offset, int length) {
        if (length == 0) {
            return EMPTY;
        }
        ByteOrder order = bb.order();
        bb = bb.duplicate();
        bb.position(offset);
        if (length > 0) {
            bb.limit(offset + length);
        }
        ByteBuffer bbSlice = bb.slice();
        bbSlice.order(order);
        return bbSlice;
    }

    public static ByteBuffer getSlice(ByteBuffer bb, int offset) {
        return ByteBufferUtils.getSlice(bb, offset, -1);
    }

    public static ByteBuffer concat(List<ByteBuffer> bbs) {
        long length = 0L;
        for (ByteBuffer bb : bbs) {
            bb.rewind();
            length += (long)bb.remaining();
        }
        if (length > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("Buffers are too large for concatenation");
        }
        if (length == 0L) {
            return EMPTY;
        }
        ByteBuffer bbNew = ByteBuffer.allocateDirect((int)length);
        for (ByteBuffer bb : bbs) {
            bb.rewind();
            bbNew.put(bb);
        }
        bbNew.rewind();
        return bbNew;
    }

    public static ByteBuffer concat(ByteBuffer ... bb) {
        return ByteBufferUtils.concat(Arrays.asList(bb));
    }

    public static ByteBuffer copy(ByteBuffer bb) {
        int capacity = bb.limit();
        int pos = bb.position();
        bb.rewind();
        ByteBuffer copy = bb.isDirect() ? ByteBuffer.allocateDirect(capacity) : ByteBuffer.allocate(capacity);
        copy.order(bb.order());
        copy.put(bb);
        copy.position(pos);
        bb.position(pos);
        return copy;
    }
}

